/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2005 by Myricom, Inc.  All rights reserved.                 *
 *************************************************************************/

#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx__lib_types.h"
#include "mx__fops.h"
#include "mx_raw.h"
#include "mx__driver_interface.h"
#include "mcp_config.h"

#define MX_RAW_NUM_TRANSMITS (MX_MCP_KREQQ_CNT - 2) /* XXX FIXME XXX */

struct mx_raw_endpoint
{
  MX__MUTEX_T lock;
  MX__MUTEX_T next_event_lock;
  mx_endpt_handle_t handle;
  int board_num;
  int board_type;
  int pending_sends;
};

MX_FUNC(mx_endpt_handle_t)
mx_raw_handle(mx_raw_endpoint_t ep)
{
  return ep->handle;
}


MX_FUNC(mx_return_t)
mx_raw_open_endpoint(uint32_t board_number,
		     mx_param_t *params_array,
		     uint32_t params_count,
		     mx_raw_endpoint_t *endpoint)

{
  struct mx_raw_endpoint *ep;
  mx_get_board_val_t board_type;
  mx_endpt_handle_t handle;
  int rc;

  ep = mx_calloc(1, sizeof(*ep));
  if (ep == NULL) {
    rc = MX_NO_RESOURCES;
    goto abort_with_nothing;
  }

  rc = mx__open(board_number, -2, &handle);
  if (rc != MX_SUCCESS)
    goto abort_with_ep;

  rc = mx__set_raw(handle);
  if (rc != 0) {
    goto abort_with_handle;
  }

  ep->handle = handle;
  ep->board_num = board_number;

  /* get board type */
  board_type.board_number = board_number;
  rc = mx__get_board_type(ep->handle, &board_type);
  if (rc) {
    goto abort_with_handle;
  }
  ep->board_type = board_type.val;

  MX__MUTEX_INIT(&ep->lock);
  MX__MUTEX_INIT(&ep->next_event_lock);

  *endpoint = ep;
  return rc;

 abort_with_handle:
  mx__close(handle);

 abort_with_ep:
  mx_free(ep);

 abort_with_nothing:
  return rc;

}

MX_FUNC(mx_return_t)
mx_raw_close_endpoint(mx_raw_endpoint_t endpoint)
{

  MX__MUTEX_DESTROY(&endpoint->lock);
  MX__MUTEX_DESTROY(&endpoint->next_event_lock);
  mx_free(endpoint);

  return MX_SUCCESS;
}


MX_FUNC(mx_return_t)
mx_raw_send(mx_raw_endpoint_t endpoint,
	    uint32_t physical_port,
	    void *route_pointer,
	    uint32_t route_length,
	    void *send_buffer,
	    uint32_t buffer_length,
	    void *context)
{
  mx_raw_send_t x;
  int rc;
  
  MX__MUTEX_LOCK(&endpoint->lock);
  if (endpoint->pending_sends >= MX_RAW_NUM_TRANSMITS) {
    rc = MX_NO_RESOURCES;
    goto abort;
  }

  /* build the raw send */
  x.physical_port = physical_port;
  x.route_pointer = (uintptr_t)route_pointer;
  x.route_length = route_length;
  x.data_pointer = (uintptr_t)send_buffer;
  x.buffer_length = buffer_length;
  x.context = (uintptr_t)context;
  /*  printf("0x%"PRIx64" submitted\n", x.context);*/
  rc = mx__raw_send(endpoint->handle, &x);
  if (rc == MX_SUCCESS)
    endpoint->pending_sends++;

  abort:
  MX__MUTEX_UNLOCK(&endpoint->lock);
  return rc;
}

MX_FUNC(mx_return_t)
mx_raw_next_event(mx_raw_endpoint_t endpoint,
		  uint32_t *incoming_port,
		  void **context,
		  void *recv_buffer,
		  uint32_t *recv_bytes,
		  uint32_t timeout_ms,
		  mx_raw_status_t *status)
{
  mx_raw_next_event_t e;
  int rc;

  e.timeout = timeout_ms;
  e.recv_buffer = (uint64_t)(uintptr_t)recv_buffer;
  e.recv_bytes = *recv_bytes;
  MX__MUTEX_LOCK(&endpoint->next_event_lock);
  rc = mx__raw_get_next_event(endpoint->handle, &e);
  MX__MUTEX_UNLOCK(&endpoint->next_event_lock);
  if (rc != 0) {
    goto abort;
  }

  *status = e.status;
  switch (e.status) {
  case MX_RAW_NO_EVENT:
    /* nothing to do here, move along... */
    break;
  case MX_RAW_SEND_COMPLETE:
    *context = (void *)(uintptr_t)e.context;
    /*    printf("0x%"PRIx64" completed\n", e.context);*/
    MX__MUTEX_LOCK(&endpoint->lock);
    endpoint->pending_sends--;
    MX__MUTEX_UNLOCK(&endpoint->lock);
    break;
  case MX_RAW_RECV_COMPLETE:
    *incoming_port = e.incoming_port;
    *recv_bytes = e.recv_bytes;
    break;
  default:
    rc = MX_NIC_DEAD;
    break;
  }

 abort:
  return rc;
}

#if MX_OS_WINNT
MX_FUNC(mx_return_t)
mx_raw_enable_hires_timer(mx_raw_endpoint_t endpoint)
{
  return MX_FAILURE;
}

MX_FUNC(mx_return_t)
mx_raw_disable_hires_timer(mx_raw_endpoint_t endpoint)
{
  return MX_FAILURE;
}
#endif

MX_FUNC(mx_return_t)
mx_raw_set_route_begin(mx_raw_endpoint_t endpoint)
{

  int rc;

  MX__MUTEX_LOCK(&endpoint->lock);
  rc = mx__set_route_begin(endpoint->handle, endpoint->board_num);
  MX__MUTEX_UNLOCK(&endpoint->lock);
  return rc;
}

MX_FUNC(mx_return_t)
mx_raw_set_route_end(mx_raw_endpoint_t endpoint)
{
  int rc;


  MX__MUTEX_LOCK(&endpoint->lock);
  rc = mx__set_route_end(endpoint->handle, endpoint->board_num);
  MX__MUTEX_UNLOCK(&endpoint->lock);
  return rc;

}

MX_FUNC(mx_return_t)
mx_raw_set_route(mx_raw_endpoint_t endpoint,
		 uint64_t destination_id,
		 void *route,
		 uint32_t route_length,
		 uint32_t destination_port,
		 uint32_t source_port,
		 mx_host_type_t host_type)
{
  mx_set_route_t s;
  int rc;

  s.mac_low32 = (uint32_t)destination_id;
  s.mac_high16 = (uint16_t)(destination_id >> 32);
  s.route_pointer = (uint64_t)(uintptr_t)route;
  s.source_port = source_port;
  s.route_length = route_length;
  s.host_type = (uint32_t)host_type;
  MX__MUTEX_LOCK(&endpoint->lock);
  rc = mx__set_route(endpoint->handle, &s);
  MX__MUTEX_UNLOCK(&endpoint->lock);
  return rc;

}


MX_FUNC(mx_return_t)
mx_raw_clear_routes(mx_raw_endpoint_t endpoint,
		    uint64_t destination_id,
		    uint32_t port)
{
  mx_set_route_t s;
  int rc;

  s.mac_low32 = (uint32_t)destination_id;
  s.mac_high16 = (uint16_t)(destination_id >> 32);
  s.source_port = port;

  MX__MUTEX_LOCK(&endpoint->lock);
  rc = mx__raw_clear_routes(endpoint->handle, &s);
  MX__MUTEX_UNLOCK(&endpoint->lock);
  return rc;
}

MX_FUNC(mx_return_t)
mx_raw_remove_peer(mx_raw_endpoint_t endpoint,
		   uint64_t destination_id)
{
  mx_raw_destination_t d;
  int rc;

  d.mac_low32 = (uint32_t)destination_id;
  d.mac_high16 = (uint16_t)(destination_id >> 32);

  MX__MUTEX_LOCK(&endpoint->lock);
  rc = mx__remove_peer(endpoint->handle, &d);
  MX__MUTEX_UNLOCK(&endpoint->lock);
  return rc;
}

static void
mx_nic_id_to_macaddr(uint64_t nic_id, uint8_t macaddr[6])
{
  int i;

  for (i = 0; i < 6; i++) {
    macaddr[5 - i] = (uint8_t)((nic_id >> (i * 8)) & 0xff);
  }
  
}


MX_FUNC(mx_return_t)
mx_raw_set_map_version(mx_raw_endpoint_t endpoint,
		       uint32_t physical_port,
		       uint64_t mapper_id,
		       uint32_t map_version,
		       uint32_t num_nodes,
		       uint32_t mapping_complete)
{
  mx_mapper_state_t s;
  int rc;

  mx_nic_id_to_macaddr(mapper_id, s.mapper_mac);
  s.iport = physical_port;
  s.map_version = map_version;
  s.num_hosts = num_nodes;
  s.network_configured = mapping_complete;
  s.level = 0;
  s.flags = 0;
#if MX_VALGRIND_DEBUG
  s.board_number = 0;
  s.routes_valid = 1;
#endif
  MX__MUTEX_LOCK(&endpoint->lock);
  rc = mx__set_mapper_state(endpoint->handle, &s);
  MX__MUTEX_UNLOCK(&endpoint->lock);
  return rc;
}


MX_FUNC(mx_return_t)
mx_raw_line_speed(mx_raw_endpoint_t endpoint, mx_line_speed_t *speed)
{
  switch (endpoint->board_type) {
    case MX_BOARD_TYPE_D:
    case MX_BOARD_TYPE_E:
      *speed = MX_SPEED_2G;
      break;
    case MX_BOARD_TYPE_Z:
      *speed = MX_SPEED_10G;
      break;
    default:
      return MX_BAD_BAD_BAD;
  }
  return MX_SUCCESS;
}

MX_FUNC(mx_return_t)
mx_raw_num_ports(mx_raw_endpoint_t endpoint, uint32_t *num_ports)
{
  
  uint32_t io;
  int rc;

  io = endpoint->board_num;
  rc = mx__get_num_ports(endpoint->handle, &io);
  if (rc != 0)
    return rc;
  *num_ports = io;
  return MX_SUCCESS;
}

MX_FUNC(mx_return_t)
mx_raw_set_hostname(mx_raw_endpoint_t endpoint, char *hostname)
{
  mx_set_hostname_t h;

  h.len = (uint32_t)strlen(hostname) + 1;  /* need to account for null */
  h.va = (uint64_t)(uintptr_t)hostname;
  h.board_number = endpoint->board_num;
  return mx__set_host_name(endpoint->handle, &h);
}

#if MX_DRIVER_API_MAGIC >= 0x500
MX_FUNC(mx_return_t)
mx_raw_set_nic_reply_info(mx_raw_endpoint_t endpoint, void *blob, uint32_t len)
{
  mx_raw_set_nic_reply_info_t r;

  r.len = len;
  r.blob_va = (uint64_t)(uintptr_t)blob;
  return mx__raw_set_nic_reply_info(endpoint->handle, &r);
}

#endif
